/* -*- coding: utf-8 -*-
 *
 * Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License (version 2) as
 *  published by the Free Software Foundation.  It is distributed in the
 *  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 * $Id$
 *
 */

var Bullet = function(size, color, frame, w, h, x, y, dir, speed) {
    this.size  = size;
    this.color = color;
    this.frame = frame;
    this.w     = w;
    this.h     = h;
    this.x     = x;
    this.y     = y;
    this.dir   = dir;
    this.speed = speed;

    this.getSize = function() {
        return this.size;
    };
    this.getColor = function() {
        return this.color;
    };
    this.next = function() {
        var pos = this.calcNext();
        this.x = pos[0];
        this.y = pos[1];
    };
    this.vanished = function(width, height) {
        return (width < this.x || height < this.y || this.x < 0 || this.y < 0);
    };
    this.draw = function(ctx) {
        if (this.frame) {
            var pos = this.calcNext();
            var dir = Math.atan2(pos[0] - this.x, this.y - pos[1]);

            ctx.save();

            if (this.frame.style == "rect") {
                ctx.beginPath();
                ctx.strokeStyle = this.frame.color;
                ctx.translate(this.x, this.y);
                ctx.rotate(dir);
                ctx.rect(-this.frame.width / 2, -this.frame.height / 2,
                         this.frame.width, this.frame.height);
                ctx.stroke();
                ctx.closePath();
            }

            ctx.restore();
        }

        ctx.beginPath();
        ctx.fillStyle = this.getColor();
        ctx.arc(this.x, this.y, this.getSize(), 0, Math.PI * 2.0, true);
        ctx.fill();
        ctx.closePath();
    };
};

var LinerBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new Bullet(size, color, frame, w, h, x, y, dir, speed);

    that.deltaBaseX = Math.cos(that.dir);
    that.deltaBaseY = Math.sin(that.dir);

    that.calcNext = function() {
        return [this.x + this.getDeltaX(), this.y + this.getDeltaY()];
    };
    that.getDeltaX = function() {
        return this.speed * this.deltaBaseX;
    };
    that.getDeltaY = function() {
        return this.speed * this.deltaBaseY;
    };

    return that;
};

var XReflectLinerBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that  = new LinerBullet(size, color, frame, w, h, x, y, dir, speed);
    var count = 0;

    that.calcNext = function() {
        var x = this.x + this.getDeltaX();
        var y = this.y + this.getDeltaY();

        if ((0 >= x || x >= this.w) && count < 5) {
            count++;
            this.dir += (2 - this.dir) - (this.dir - 1);
            this.deltaBaseX = Math.cos(this.dir);
            this.deltaBaseY = Math.sin(this.dir);
            x = this.x + this.getDeltaX();
            y = this.y + this.getDeltaY();
        }

        return [x, y];
    };

    return that;
}

var AxisExtendBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new LinerBullet(size, color, frame, w, h, x, y, dir, speed);

    that.dx = that.speed * that.deltaBaseX / 1.5;
    that.dy = that.speed * that.deltaBaseY / 1.5;

    that.getDeltaX = function() {
        return this.dx * this.getAxisDeltaX();
    };
    that.getDeltaY = function() {
        return this.dy * this.getAxisDeltaY();
    };
    that.getCurrentDelta = function(count, thresh) {
        var c = count / this.speed;
        return c > thresh ? thresh : c;
    }

    return that;
};

var XYExtendBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new AxisExtendBullet(size, color, frame, w, h, x, y, dir, speed);

    that.i = 1;
    that.getAxisDeltaX = function() {
        return this.getCurrentDelta(this.i, 1.5);
    };
    that.getAxisDeltaY = function() {
        return this.getCurrentDelta(this.i++, 1.5);
    };

    return that;
};

var XExtendBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new AxisExtendBullet(size, color, frame, w, h, x, y, dir, speed);

    that.i = 1;
    that.getAxisDeltaX = function() {
        return this.getCurrentDelta(this.i, 1.5);
    };
    that.getAxisDeltaY = function() {
        return 1 / this.getCurrentDelta(this.i++, 1.5);
    };

    return that;
};

var YExtendBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new AxisExtendBullet(size, color, frame, w, h, x, y, dir, speed);

    that.i = 1;
    that.getAxisDeltaX = function() {
        return 1 / this.getCurrentDelta(this.i, 2);
    };
    that.getAxisDeltaY = function() {
        return this.getCurrentDelta(this.i++, 2);
    };

    return that;
};

var CurveBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new Bullet(size, color, frame, w, h, x, y, dir, speed);

    that.delta = 1 / (that.speed * 100);
    that.i     = 1;

    that.getDir = function() {
        return this.dir;
    };
    that.calcNext = function() {
        var d = this.getDir();
        var x = this.x + Math.cos(d) * this.i;
        var y = this.y + Math.sin(d) * this.i;
        this.dir += (this.delta / this.i) * this.getSign();
        this.i   += (this.delta / this.i);
        return [x, y];
    };

    return that;
};

var LeftCurveBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new CurveBullet(size, color, frame, w, h, x, y, dir, speed);

    that.getSign = function() {
        return 1;
    };

    return that;
};

var RightCurveBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that = new CurveBullet(size, color, frame, w, h, x, y, dir, speed);

    that.getSign = function() {
        return -1;
    };

    return that;
};

var XReflectLeftCurveBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that  = new LeftCurveBullet(size, color, frame, w, h, x, y, dir, speed);
    var count = 0;

    that.getDir = function() {
        var x = this.x + Math.cos(this.dir) * this.i;
        var y = this.y + Math.sin(this.dir) * this.i;

        if ((0 >= x || x >= this.w) && count < 5) {
            count++;
            this.dir += (2 - this.dir) - (this.dir - 1);
        }

        return this.dir;
    };

    return that;
};

var XReflectRightCurveBullet = function(size, color, frame, w, h, x, y, dir, speed) {
    var that  = new RightCurveBullet(size, color, frame, w, h, x, y, dir, speed);
    var count = 0;

    that.getDir = function() {
        var x = this.x + Math.cos(this.dir) * this.i;
        var y = this.y + Math.sin(this.dir) * this.i;

        if ((0 >= x || x >= this.w) && count < 5) {
            count++;
            this.dir += (2 - this.dir) - (this.dir - 1);
        }

        return this.dir;
    };

    return that;
};

var EnemyBullets = [LinerBullet,
                    XReflectLinerBullet,
                    XYExtendBullet,
                    XExtendBullet,
                    YExtendBullet,
                    LeftCurveBullet,
                    RightCurveBullet,
                    XReflectLeftCurveBullet,
                    XReflectRightCurveBullet];
