/* -*- coding: utf-8 -*-
 *
 * Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License (version 2) as
 *  published by the Free Software Foundation.  It is distributed in the
 *  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 * $Id$
 *
 */

function touch(x1, y1, s1, x2, y2, s2) {
    var dx = Math.pow(x2 - x1, 2);
    var dy = Math.pow(y2 - y1, 2);
    if (Math.sqrt(dx + dy) < s1 + s2)
        return true;
    return false;
}

var Bomb = function(x, y, w, h, size, speed, colors) {
    this.x      = x;
    this.y      = y;
    this.w      = w;
    this.h      = h;
    this.size   = size;
    this.speed  = speed;
    this.colors = colors;
    this.colIdx = 0;

    this.update = function(enemies) {
        this.size += this.speed;

        for (var ei = 0; ei < enemies.length; ei++) {
            var enemy  = enemies[ei];
            var delIdx = new Array();

            // judge bullet
            for (var i = 0; i < enemy.bullets.length; i++) {
                if (touch(this.x, this.y, this.size,
                          enemy.bullets[i].x, enemy.bullets[i].y, enemy.bullets[i].size))
                    delIdx.push(i);
            }
            enemy.delBullet(delIdx);

            // judge enemy
            if (touch(this.x, this.y, this.size, enemy.x, enemy.y, enemy.size) && enemy.life > 0)
                enemy.life--;
        }
    };

    this.draw = function(ctx) {
        ctx.beginPath();
        ctx.fillStyle = this.colors[this.colIdx];
        ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2.0, true);
        ctx.fill();
        ctx.closePath();

        if (++this.colIdx >= this.colors.length)
            this.colIdx = 0;
    };

    this.vanished = function() {
        if (this.size > this.w && this.size > this.h)
            return true;
        return false;
    };
};

var Trooper = function(name, actList, image, size, hitSize, color, hitColor,
                       x, y, w, h, life, speed, numBombs, bombColors, barrages) {

    this.bullets    = new Array();
    this.callbacks  = {};
    this.bomb       = null;
    this.name       = name;
    this.actList    = actList;
    this.image      = image;
    this.size       = size;
    this.hitSize    = hitSize;
    this.color      = color;
    this.hitColor   = hitColor;
    this.x          = x;
    this.y          = y;
    this.w          = w;
    this.h          = h;
    this.life       = life;
    this.maxLife    = life;
    this.speed      = speed;
    this.numBombs   = numBombs;
    this.bombColors = bombColors;
    this.barrages   = barrages;

    this.move = function(x, y, area) {
        this.x += x;
        this.y += y;
        this.adjustPosition(area);
    };

    this.moveAbs = function(x, y, area) {
        this.x = x;
        this.y = y;
        this.adjustPosition(area);
    };

    this.adjustPosition = function(area) {
        if (area == null)
            area = 1;

        if (this.x > this.w)
            this.x = this.w;

        if (this.y > this.h / area)
            this.y = this.h / area;

        if (this.x < 0)
            this.x = 0;

        if (this.y < 0)
            this.y = 0;
    };

    this.isDead = function() {
        return this.life <= 0;
    };

    this.update = function(enemies) {
        // update my action
        this.actList.update(this, enemies);

        // update bomb
        if (this.bomb) {
            this.bomb.update(enemies);
            if (this.bomb.vanished())
                this.bomb = null;
        }

        // update/delete my bullets
        {
            var delIdx = new Array();
            for (var i = 0; i < this.bullets.length; i++) {
                this.bullets[i].next();

                if (this.bullets[i].vanished(this.w, this.h))
                    delIdx.push(i);
            }
            this.delBullet(delIdx);
        }

        // update/delete enemies' bullets
        {
            var hit = false;

            for (var ei = 0; ei < enemies.length; ei++) {
                var enemy  = enemies[ei];
                var delIdx = new Array();

                // judge/delete hit bullets
                for (var i = 0; i < enemy.bullets.length; i++) {
                    if (touch(this.x, this.y, this.hitSize,
                              enemy.bullets[i].x, enemy.bullets[i].y, enemy.bullets[i].size))
                        delIdx.push(i);
                }

                enemy.delBullet(delIdx);

                if (delIdx.length > 0)
                    hit = true;
            }

            // update my life
            if (hit && this.life > 0) {
                this.life--;

                if (this.callbacks["damaged"])
                    this.callbacks["damaged"](this);
            }
        }
    };

    this.draw = function(ctx) {
        // draw bomb
        if (this.bomb)
            this.bomb.draw(ctx);

        // draw bullets
        for (var i = 0; i < this.bullets.length; i++)
            this.bullets[i].draw(ctx);

        // draw trooper
        if (this.image) {
            var w = this.image.width * (this.size / this.image.height);
            var h = this.size;
            var x = this.x - (w / 2);
            var y = this.y - (h / 2);
            ctx.drawImage(this.image, x, y, w, h);
        }
        else {
            ctx.beginPath();
            ctx.fillStyle = this.color;
            ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2.0, true);
            ctx.fill();
            ctx.closePath();
        }

        // draw hit area
        if (this.color != this.hitColor || this.size != this.hitSize) {
            ctx.beginPath();
            ctx.fillStyle = this.hitColor;
            ctx.arc(this.x, this.y, this.hitSize, 0, Math.PI * 2.0, true);
            ctx.fill();
            ctx.closePath();
        }
    };

    this.addBullet = function(bulletType, size, color, frame, x, y, dir, speed) {
        this.bullets.push(
            new bulletType(size, color, frame, this.w, this.h, this.x + x, this.y + y, dir, speed));

        if (this.callbacks["addBullet"])
            this.callbacks["addBullet"](this);
    };

    this.delBullet = function(idx) {
        if (idx.length <= 0)
            return;

        var newBullets = new Array();

        for (var i = 0; i < idx.length; i++)
            this.bullets[idx[i]] = null;

        for (var i = 0; i < this.bullets.length; i++)
            if (this.bullets[i])
                newBullets.push(this.bullets[i]);

        this.bullets = newBullets;
    };

    this.clearBullet = function() {
        this.bullets = new Array();
    };

    this.addBomb = function() {
        if (!this.bomb && this.numBombs > 0) {
            this.bomb = new Bomb(this.x, this.y, this.w, this.h,
                                 this.size * 2, 10, this.bombColors);
            this.numBombs--;

            if (this.callbacks["addBomb"])
                this.callbacks["addBomb"](this);
        }
    };

    this.registerCallback = function(name, callback) {
        this.callbacks[name] = callback;
    };
}
