/*
 * ksyslog: In-kernel syslog receiver 
 * Copyright(C) 2013 Atzm WATANABE All rights reserved
 * Distributed under the GPL
 */

#ifndef KSYSLOG_H
#define KSYSLOG_H

#define UDP_ENCAP_KSYSLOG 16

enum ksyslog_facility {
	KSYSLOG_F_KERN,
	KSYSLOG_F_USER,
	KSYSLOG_F_MAIL,
	KSYSLOG_F_DAEMON,
	KSYSLOG_F_AUTH,
	KSYSLOG_F_SYSLOG,
	KSYSLOG_F_LPR,
	KSYSLOG_F_NEWS,
	KSYSLOG_F_UUCP,
	KSYSLOG_F_CRON,
	KSYSLOG_F_AUTHPRIV,
	KSYSLOG_F_FTP,
	KSYSLOG_F_NTP,
	KSYSLOG_F_AUDIT,
	KSYSLOG_F_ALERT,
	KSYSLOG_F_CRONPRIV,  /* ? */
	KSYSLOG_F_LOCAL0,
	KSYSLOG_F_LOCAL1,
	KSYSLOG_F_LOCAL2,
	KSYSLOG_F_LOCAL3,
	KSYSLOG_F_LOCAL4,
	KSYSLOG_F_LOCAL5,
	KSYSLOG_F_LOCAL6,
	KSYSLOG_F_LOCAL7,
	__KSYSLOG_F_MAX,
};

enum ksyslog_severity {
	KSYSLOG_S_EMERG,
	KSYSLOG_S_ALERT,
	KSYSLOG_S_CRIT,
	KSYSLOG_S_ERR,
	KSYSLOG_S_WARN,
	KSYSLOG_S_NOTICE,
	KSYSLOG_S_INFO,
	KSYSLOG_S_DEBUG,
	__KSYSLOG_S_MAX,
};

struct ksyslog_entry {
	struct list_head list;
	struct timeval tv;

	unsigned long time;
	unsigned int priority;
	enum ksyslog_facility facility;
	enum ksyslog_severity severity;

	union {
		__u8 addr8[4];
		__be32 addr32;
	} daddr, saddr;

	__be16 dport, sport;

	size_t length;
	char *data;

	struct rcu_head rcu;
};

struct ksyslog_queue {
	struct list_head head;
	atomic64_t nr_queued;
	atomic64_t nr_written;
	atomic64_t nr_dropped;
};

static inline const char *
ksyslog_facility_str(const enum ksyslog_facility facility)
{
	switch (facility) {
	case KSYSLOG_F_KERN:     return "kern";
	case KSYSLOG_F_USER:     return "user";
	case KSYSLOG_F_MAIL:     return "mail";
	case KSYSLOG_F_DAEMON:   return "daemon";
	case KSYSLOG_F_AUTH:     return "auth";
	case KSYSLOG_F_SYSLOG:   return "syslog";
	case KSYSLOG_F_LPR:      return "lpr";
	case KSYSLOG_F_NEWS:     return "news";
	case KSYSLOG_F_UUCP:     return "uucp";
	case KSYSLOG_F_CRON:     return "cron";
	case KSYSLOG_F_AUTHPRIV: return "authpriv";
	case KSYSLOG_F_FTP:      return "ftp";
	case KSYSLOG_F_NTP:      return "ntp";
	case KSYSLOG_F_AUDIT:    return "audit";
	case KSYSLOG_F_ALERT:    return "alert";
	case KSYSLOG_F_CRONPRIV: return "cronpriv";
	case KSYSLOG_F_LOCAL0:   return "local0";
	case KSYSLOG_F_LOCAL1:   return "local1";
	case KSYSLOG_F_LOCAL2:   return "local2";
	case KSYSLOG_F_LOCAL3:   return "local3";
	case KSYSLOG_F_LOCAL4:   return "local4";
	case KSYSLOG_F_LOCAL5:   return "local5";
	case KSYSLOG_F_LOCAL6:   return "local6";
	case KSYSLOG_F_LOCAL7:   return "local7";
	default:                 return "unknown";
	}
}

static inline const char *
ksyslog_severity_str(const enum ksyslog_severity severity)
{
	switch (severity) {
	case KSYSLOG_S_EMERG:  return "emerg";
	case KSYSLOG_S_ALERT:  return "alert";
	case KSYSLOG_S_CRIT:   return "crit";
	case KSYSLOG_S_ERR:    return "err";
	case KSYSLOG_S_WARN:   return "warn";
	case KSYSLOG_S_NOTICE: return "notice";
	case KSYSLOG_S_INFO:   return "info";
	case KSYSLOG_S_DEBUG:  return "debug";
	default:               return "unknown";
	}
}

static inline unsigned int
ksyslog_month_num(const char *month)
{
	if (!memcmp(month, "Jan", 3))
		return 1;
	else if (!memcmp(month, "Feb", 3))
		return 2;
	else if (!memcmp(month, "Mar", 3))
		return 3;
	else if (!memcmp(month, "Apr", 3))
		return 4;
	else if (!memcmp(month, "May", 3))
		return 5;
	else if (!memcmp(month, "Jun", 3))
		return 6;
	else if (!memcmp(month, "Jul", 3))
		return 7;
	else if (!memcmp(month, "Aug", 3))
		return 8;
	else if (!memcmp(month, "Sep", 3))
		return 9;
	else if (!memcmp(month, "Oct", 3))
		return 10;
	else if (!memcmp(month, "Nov", 3))
		return 11;
	else if (!memcmp(month, "Dec", 3))
		return 12;
	else
		return 0;
}

#endif
