/* -*- coding: utf-8 -*-
 *
 * Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License (version 2) as
 *  published by the Free Software Foundation.  It is distributed in the
 *  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 * $Id$
 *
 */

var Bullet = function(size, color, x, y, dir, speed) {
    this.size  = size;
    this.color = color;
    this.x     = x;
    this.y     = y;
    this.dir   = dir;
    this.speed = speed;

    this.draw = function(ctx) {
        ctx.beginPath();
        ctx.fillStyle = this.color;
        ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2.0, true);
        ctx.fill();
        ctx.closePath();
    };
    this.vanished = function(width, height) {
        return (width < this.x || height < this.y || this.x < 0 || this.y < 0);
    };
};

var LinerBullet = function(size, color, x, y, dir, speed) {
    var that = new Bullet(size, color, x, y, dir, speed);

    that.deltaBaseX = Math.cos(that.dir);
    that.deltaBaseY = Math.sin(that.dir);

    that.next = function() {
        this.x += this.getDeltaX();
        this.y += this.getDeltaY();
    };
    that.getDeltaX = function() {
        return this.speed * this.deltaBaseX;
    };
    that.getDeltaY = function() {
        return this.speed * this.deltaBaseY;
    };

    return that;
};

var AxisExtendBullet = function(size, color, x, y, dir, speed) {
    var that = new LinerBullet(size, color, x, y, dir, speed);

    that.dx = that.speed * that.deltaBaseX / 2;
    that.dy = that.speed * that.deltaBaseY / 2;

    that.getDeltaX = function() {
        return this.dx * this.getAxisDeltaX();
    };
    that.getDeltaY = function() {
        return this.dy * this.getAxisDeltaY();
    };
    that.getCurrentDelta = function(count, thresh) {
        var c = count / this.speed;
        return c > thresh ? thresh : c;
    }

    return that;
};

var XYExtendBullet = function(size, color, x, y, dir, speed) {
    var that = new AxisExtendBullet(size, color, x, y, dir, speed);

    that.i = 1;
    that.getAxisDeltaX = function() {
        return this.getCurrentDelta(this.i, 2);
    };
    that.getAxisDeltaY = function() {
        return this.getCurrentDelta(this.i++, 2);
    };

    return that;
};

var XExtendBullet = function(size, color, x, y, dir, speed) {
    var that = new AxisExtendBullet(size, color, x, y, dir, speed);

    that.i = 1;
    that.getAxisDeltaX = function() {
        return this.getCurrentDelta(this.i, 2);
    };
    that.getAxisDeltaY = function() {
        return 1 / this.getCurrentDelta(this.i++, 2);
    };

    return that;
};

var YExtendBullet = function(size, color, x, y, dir, speed) {
    var that = new AxisExtendBullet(size, color, x, y, dir, speed);

    that.i = 1;
    that.getAxisDeltaX = function() {
        return 1 / this.getCurrentDelta(this.i, 2);
    };
    that.getAxisDeltaY = function() {
        return this.getCurrentDelta(this.i++, 2);
    };

    return that;
};

var CurveBullet = function(size, color, x, y, dir, speed) {
    var that = new Bullet(size, color, x, y, dir, speed);

    that.delta = 1 / (that.speed * 100);
    that.i     = 1;

    that.next = function() {
        this.x   += Math.cos(this.dir) * this.i * this.getSign();
        this.y   += Math.sin(this.dir) * this.i;
        this.dir += (this.delta / this.i);
        this.i   += (this.delta / this.i);
    };

    return that;
};

var LeftCurveBullet = function(size, color, x, y, dir, speed) {
    var that = new CurveBullet(size, color, x, y, dir, speed);

    that.getSign = function() {
        return 1;
    };

    return that;
};

var RightCurveBullet = function(size, color, x, y, dir, speed) {
    var that = new CurveBullet(size, color, x, y, dir, speed);

    that.getSign = function() {
        return -1;
    };

    return that;
};


var EnemyBullets = [LinerBullet,
                    XYExtendBullet,
                    XExtendBullet,
                    YExtendBullet,
                    LeftCurveBullet,
                    RightCurveBullet];
