/* -*- coding: utf-8 -*-
 *
 * Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License (version 2) as
 *  published by the Free Software Foundation.  It is distributed in the
 *  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 * $Id$
 *
 */

System = {
    "screen": {
        "canvas": null,
        "ctx":    null,
        "width":  0,
        "height": 0
    },
    "boss":             null,
    "player":           null,
    "backgroundObject": new Array(),
    "mainIntervalId":   0
};


/*
 *  Utility Functions
 */
function updateBackground(ctx, width, height, size, color, max) {
    if (System.backgroundObject.length < max) {
        var x = Math.ceil(Math.random() * width);
        var s = Math.ceil(Math.random() * 5);
        System.backgroundObject.push(
            new LinerBullet(size, color, x, 0, 0.5 * Math.PI, s));
    }

    var newObjs = new Array();
    for (i = 0; i < System.backgroundObject.length; i++) {
        System.backgroundObject[i].next();
        System.backgroundObject[i].draw(ctx);
        if (!System.backgroundObject[i].vanished(width, height))
            newObjs.push(System.backgroundObject[i]);
    }
    System.backgroundObject = newObjs;
}

function drawScreen(ctx, op, style, width, height) {
    var c = ctx.globalCompositeOperation;
    ctx.globalCompositeOperation = op;
    ctx.beginPath();
    ctx.fillStyle = style;
    ctx.fillRect(0, 0, width, height);
    ctx.fill();
    ctx.closePath();
    ctx.globalCompositeOperation = c;
}

function drawLifeGauge(ctx, op, trooper, x, y, width) {
    var length = trooper.life;

    if (length > width - 20)
        length = width - 20;

    var c = ctx.globalCompositeOperation;
    ctx.globalCompositeOperation = op;
    ctx.beginPath();
    ctx.fillStyle = trooper.color;
    ctx.fillRect(x, y, length, 10);
    ctx.fill();
    ctx.closePath();
    ctx.globalCompositeOperation = c;

    drawString(ctx, op, trooper.life, x + 2, y + 8, trooper.color,
               "6pt monospace", "left");
}

function drawString(ctx, op, string, x, y, color, font, align) {
    var a = ctx.textAlign;
    var f = ctx.font;
    var c = ctx.globalCompositeOperation;
    ctx.globalCompositeOperation = op;
    ctx.beginPath();
    ctx.textAlign = align;
    ctx.fillStyle = color;
    ctx.font      = font;
    ctx.fillText(string, x, y);
    ctx.fill();
    ctx.textAlign = a;
    ctx.font      = f;
    ctx.closePath();
    ctx.globalCompositeOperation = c;
}


/*
 *  Main loop
 */
function mainLoop() {
    // clear screen
    drawScreen(
        System.screen.ctx,
        "source-over",
        "rgba(8,8,8,0.5)",
        System.screen.width,
        System.screen.height
    );

    // update background objects
    updateBackground(
        System.screen.ctx,
        System.screen.width,
        System.screen.height,
        1, "#CAF", 10
    );

    // update troopers
    System.player.update(System.boss);
    System.boss.update(System.player);

    // draw troopers
    System.player.draw(System.screen.ctx);
    System.boss.draw(System.screen.ctx);

    // draw player name/life
    drawString(
        System.screen.ctx,
        "source-over",
        System.player.name,
        10, System.screen.height - 25,
        "#ACF", "9pt monospace", "left"
    );
    drawLifeGauge(
        System.screen.ctx,
        "lighter",
        System.player,
        10,
        System.screen.height - 20,
        System.screen.width
    );

    // draw boss name/life
    drawString(
        System.screen.ctx,
        "source-over",
        System.boss.name, 10, 35,
        "#FCA", "9pt monospace", "left"
    );
    drawLifeGauge(
        System.screen.ctx,
        "lighter",
        System.boss,
        10,
        10,
        System.screen.width
    );

    // is dead?
    if (System.player.isDead()) {
        drawString(
            System.screen.ctx, "source-over",
            "YOU LOST",
            System.screen.width / 2,
            System.screen.height / 2,
            "#FCA", "24pt monospace", "center"
        );
        clearInterval(System.mainIntervalId);
        System.mainIntervalId = 0;
    }

    if (System.boss.isDead()) {
        drawString(
            System.screen.ctx, "source-over",
            "YOU WON",
            System.screen.width / 2,
            System.screen.height / 2,
            "#ACF", "24pt monospace", "center"
        );
        clearInterval(System.mainIntervalId);
        System.mainIntervalId = 0;
    }
}


/*
 *  Initializer
 */
function initGame(canvas, bossId, bossData, playerData) {
    System.screen.canvas = canvas;
    System.screen.ctx    = System.screen.canvas.getContext("2d");
    System.screen.width  = System.screen.canvas.width;
    System.screen.height = System.screen.canvas.height;

    System.screen.ctx.globalCompositeOperation = "lighter";

    if (System.mainIntervalId) {
        clearInterval(System.mainIntervalId);
        System.mainIntervalId = 0;
    }

    drawScreen(
        System.screen.ctx,
        "source-over",
        "rgba(0,0,0,1)",
        System.screen.width,
        System.screen.height
    );

    System.player = new Trooper(
        playerData.name,
        new ActionList([new ManualAction(new ManualShot())]),
        playerData.size,
        "#33F",
        System.screen.width / 2,
        System.screen.height - System.screen.height / 7,
        System.screen.width,
        System.screen.height,
        playerData.hitpoint,
        playerData.speed,
        [new LinerBarrage(YExtendBullet,
                          playerData.shotsize,
                          "#3FF",
                          playerData.shotinterval,
                          playerData.shotspeed,
                          playerData.shotlevel,
                          -0.5),
         new LinerBarrage(YExtendBullet,
                          playerData.shotsize,
                          "#3FF",
                          playerData.shotinterval,
                          playerData.shotspeed,
                          playerData.shotlevel,
                          0.5),
         new CircularBarrage(LinerBullet,
                          playerData.shotsize,
                          "#3FF",
                          playerData.shotinterval,
                          playerData.shotspeed,
                          playerData.shotlevel + 2,
                          -0.5)]
    );

    var actList = EnemyActionLists[bossData.mtime % EnemyActionLists.length];
    var shot    = EnemyShots[bossData.hitpoint % EnemyShots.length];
    var numAct  = bossData.agility       % (EnemyActions.length  - 1) + 1;
    var numBlt  = bossData.skills.length % (EnemyBullets.length  - 1) + 1;
    var numBrrg = bossData.skills.length % (EnemyBarrages.length - 1) + 1;
    var acts    = new Array();
    var brrgs   = new Array();

    debugMsgSet("ActionsIdx:");

    for (i = 0; i < numAct; i++) {
        var idx = (bossData.agility + i) % EnemyActions.length;
        acts.push(new EnemyActions[idx](new shot()));

        debugMsgAdd(String(idx) + ",");
    }

    debugMsgAdd("<br>");

    for (i = 0; i < numBrrg; i++) {
        var idx     = (bossData.skills.length + i) % EnemyBarrages.length;
        var brrgCls = EnemyBarrages[idx];

        debugMsgAdd("BarragesIdx:" + String(idx) + " / BulletsIdx:");

        for (k = 0; k < numBlt; k++) {
            var iidx = (bossData.skills.length + i + k) % EnemyBullets.length;
            var ss   = Math.ceil(bossData.luck / 3);
            brrgs.push(
                new brrgCls(
                    EnemyBullets[iidx],
                    ss < 3 ? 3 : ss,
                    "#FF3",
                    200 * 1 / Math.log(bossData.skillpoint + 0.1),
                    Math.log(bossData.strength / 15 + 0.1),
                    Math.ceil(bossData.concentration / 5)
                )
            );

            debugMsgAdd(String(iidx) + ",");
        }

        debugMsgAdd("<br>");
    }

    System.boss = new Trooper(
        bossData.name,
        new actList(acts),
        Math.ceil(50 * (1 / bossData.defense)),
        "#F33",
        System.screen.width / 2,
        System.screen.height / 7,
        System.screen.width,
        System.screen.height,
        bossData.hitpoint,
        Math.log(bossData.agility + 0.1) * 3,
        brrgs
    );

    System.backgroundObject = new Array();
    System.mainIntervalId   = setInterval(mainLoop, 20);
}
