# -*- coding: utf-8 -*-
#
#  Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id$
#

try:
    import cStringIO as _StringIO
except ImportError:
    import StringIO as _StringIO

import random
import pycodebattler


def pformat(warriors):
    order = ['NAME', 'HP', 'SP', 'STR', 'CON', 'DEF', 'AGI', 'LCK']
    mmap = {
        'NAME': 'name',
        'HP':   'hitpoint',
        'SP':   'skillpoint',
        'STR':  'strength',
        'CON':  'concentration',
        'DEF':  'defense',
        'AGI':  'agility',
        'LCK':  'luck',
    }

    horiz = [[getattr(w, mmap[n])() for n in order] for w in warriors]
    virt = zip(*horiz)

    sepnum = (len(order) * 3) - 1
    fmt = ['|']
    for i, t in enumerate(virt):
        num = max(len(str(n)) for n in list(t) + [order[i]])
        sepnum += num
        if i == 0:
            fmt.append(' %%-%ds |' % num)
        else:
            fmt.append(' %%%ds |' % num)

    fmt = ''.join(fmt + ['\n'])
    sep = ''.join(['+', '-' * sepnum, '+', '\n'])

    io = _StringIO.StringIO()
    io.write(sep)
    io.write(fmt % tuple(order))
    io.write(sep)
    for d in horiz:
        io.write(fmt % tuple(d))
    io.write(sep)
    io.flush()

    return io.getvalue()


class Warrior:
    def __init__(self, name, hitpoint, skillpoint, strength, concentration,
                 defense, agility, luck, skill_list=[], luck_range=5000):
        for val in [hitpoint, skillpoint, strength, concentration,
                    defense, agility, luck, luck_range]:
            if int(val) <= 0:
                raise ValueError('parameter includes invalid value')

        self._name = name
        self._skill_list = {}
        self._max_hitpoint = self._hitpoint = hitpoint
        self._max_skillpoint = self._skillpoint = skillpoint
        self._strength = strength
        self._concentration = concentration
        self._defense = defense
        self._agility = agility
        self._luck = luck
        self._luck_range = luck_range

        for sk in skill_list:
            self._skill_list[sk.name()] = sk

    @classmethod
    def make(klass, name, fp, skill_list=[], luck_range=5000):
        scorer = pycodebattler.score.Scorer(fp.readlines())

        hitpoint = scorer.style_score()
        skillpoint = scorer.name_score()
        strength = scorer.complex_score()
        concentration = scorer.colon_score()
        defense = scorer.lines_score()
        agility = scorer.const_score()
        luck = luck_range // (hitpoint + skillpoint + strength +
                              concentration + defense + agility) or 1

        return klass(name, hitpoint, skillpoint, strength, concentration,
                     defense, agility, luck, skill_list, luck_range)

    def __str__(self):
        return self.name()

    def name(self):
        return self._name

    def max_hitpoint(self):
        return self._max_hitpoint

    def hitpoint(self):
        return self._hitpoint

    def max_skillpoint(self):
        return self._max_skillpoint

    def skillpoint(self):
        return self._skillpoint

    def strength(self):
        return self._strength

    def concentration(self):
        return self._concentration

    def defense(self):
        return self._defense

    def agility(self):
        return self._agility

    def luck(self):
        return self._luck

    def skill_list(self):
        return self._skill_list.keys()

    def skill(self, name):
        return self._skill_list[name]

    def lucky(self):
        r = random.randint(-256, self._luck_range)
        return r <= (self.luck() + self.agility() / 8)

    def is_injured(self):
        return self.hitpoint() <= (self.max_hitpoint() / 10)

    def is_dead(self):
        return self.hitpoint() <= 0

    def can_invoke(self, name):
        try:
            return self.skill(name).point() < self.skillpoint()
        except KeyError:
            return False

    def invoke(self, name, enemy, warriors):
        try:
            return self.skill(name).invoke(self, enemy, warriors)
        except KeyError:
            return []

    def spend(self, point):
        self._skillpoint -= point
        if self._skillpoint < 0:
            self._skillpoint = 0
        return point

    def damage_from(self, enemy):
        miss = False
        critical = False

        rgosa = enemy.strength() / 2
        gosa = random.randint(0 - rgosa, rgosa)

        if enemy.lucky():
            critical = True
            damage = (enemy.strength() + gosa) * random.randint(3, 8)
        else:
            damage = enemy.strength() + gosa - self.defense()

        if damage <= 0:
            damage = 1

        if self.lucky() and not critical:
            miss = True
            damage = 0

        return self.damage(damage), miss, critical

    def damage(self, damage):
        self._hitpoint -= damage
        if self.hitpoint() < 0:
            self._hitpoint = 0
        if self.hitpoint() > self.max_hitpoint():
            self._hitpoint = self.max_hitpoint()
        return damage

    def attack_to(self, enemy):
        return enemy.damage_from(self)
