#!/usr/bin/env python
# -*- coding: utf-8 -*-

__version__ = '$Revision$'
__author__ = 'Atzm WATANABE <sitosito@p.chan.ne.jp>'
__date__ = '$Date$'
__copyright__ = 'Copyright(C) 2006 Atzm WATANABE, all rights reserved.'
__license__ = 'Python'

import re
import sys
import random
import MeCab

from ircbot import SingleServerIRCBot
from irclib import nm_to_n

import config
config.init()

import my_amazon
my_amazon.setLocale(config.get('amazon', 'locale'))
my_amazon.setLicense(config.get('amazon', 'access_key'))

try:
	set, frozenset
except NameError:
	from sets import Set as set, ImmutableSet as frozenset

def uniq(sequence):
	return list(set(sequence))

def mecab_parse(text):
	def choice_nominal(wlist):
		res = []
		for word, wtype in wlist:
			wtypes = wtype.split('-')
			if '固有名詞' in wtypes or ('名詞' in wtypes and '一般' in wtypes):
				res.append(word)
		return res

	result = []
	tag = MeCab.Tagger('-Ochasen')
	for line in tag.parse(text).split('\n'):
		if not line or line == 'EOS':
			break
		words = line.split()
		result.append((words[0], words[-1])) # word, word-type

	result = uniq(choice_nominal(result))
	return result

class AmazonBotBase(SingleServerIRCBot):
	def __init__(self):
		_server = [(config.get('irc', 'server'), config.get('irc', 'port', 'int'))]
		_nick = config.get('bot', 'nick')

		self._silent = False
		SingleServerIRCBot.__init__(self, _server, _nick, _nick)

	def get_version(self):
		return 'AmazonBot by %s, based on python-irclib' % __author__

	def on_welcome(self, c, e):
		c.join(config.get('irc', 'channel'))
		if __debug__:
			print >> sys.stderr, '* Joined %s' % config.get('irc', 'channel')

	def on_nicknameinuse(self, c, e):
		c.nick(c.get_nickname() + '_')

	def on_privmsg(self, c, e):
		return self.on_pubmsg(c, e)

	def on_pubmsg(self, c, e):
		try:
			msg = unicode(e.arguments()[0], 'iso-2022-jp')
		except UnicodeError:
			if __debug__:
				print >> sys.stderr, 'ERR: incoming message encoding failed: %s' % e.arguments()
			return False

		self.silence(msg, c, e)
		if self._silent:
			return False

		nominals = mecab_parse(msg.encode('utf-8'))
		if not nominals:
			return False

		title, url = self.process_keyword(' '.join(nominals))
		if title and url:
			channel = e.target()

			content = unicode(config.get('bot', 'content'), 'utf-8')
			title = title

			try:
				message = ('%(content)s: %(title)s %(url)s' % locals()).encode('iso-2022-jp')
			except UnicodeError:
				if __debug__:
					print >> sys.stderr, 'ERR: my message encoding failed: '
					print >> sys.stderr, ' * content> %s' % content
					print >> sys.stderr, ' * title> %s' % title
					print >> sys.stderr, ' * url> %s' % url
				return False

			c.privmsg(channel, message)

			return True

		return False

	ACTIVE_PATTERN = re.compile(config.get('bot', 'active_pattern'))
	SILENT_PATTERN = re.compile(config.get('bot', 'silent_pattern'))
	def silence(self, msg, c, e):
		ch = e.target()
		active = self.ACTIVE_PATTERN.search(msg.encode('utf-8'))
		silent = self.SILENT_PATTERN.search(msg.encode('utf-8'))
		if __debug__:
			print >> sys.stderr, 'ACTIVE_PATTERN: %s, SILENT_PATTERN: %s' % (str(active), str(silent))

		if active:
			self._silent = False
			c.privmsg(ch, unicode(config.get('bot', 'thanks'), 'utf-8').encode('iso-2022-jp'))
		elif silent:
			self._silent = True
			c.privmsg(ch, unicode(config.get('bot', 'sorry'), 'utf-8').encode('iso-2022-jp'))

	def process_keyword(self, keyword):
		return [None, None]

class AmazonBot(AmazonBotBase):
	def __init__(self):
		AmazonBotBase.__init__(self)

	def process_keyword(self, keyword):
		try:
			data = my_amazon.searchByBlended(keyword)
			if type(data.ProductLine) is not type([]):
				data.ProductLine = [data.ProductLine]
		except my_amazon.AmazonError:
			return [None, None]

		product_line = random.choice(data.ProductLine)
		detail = random.choice(product_line.ProductInfo.Details)

		url = getattr(detail, 'URL', None)
		product_name = getattr(detail, 'ProductName', None)

		return [product_name, url]

if __name__ == '__main__':
	bot = AmazonBot()
	bot.start()
