/* -*- coding: utf-8 -*-
 *
 * Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License (version 2) as
 *  published by the Free Software Foundation.  It is distributed in the
 *  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 * $Id$
 *
 */

const ACTION_UP     = 1;
const ACTION_DOWN   = 2;
const ACTION_LEFT   = 4;
const ACTION_RIGHT  = 8;
const ACTION_SHOT   = 16;
const ACTION_DELAY  = 32;
const SWITCH_BULLET = 1;
const SWITCH_BOMB   = 2;
var   currentAction = 0;
var   currentSwitch = 0;


/*
 *  Functions
 */
function getActionByKeyCode(keyCode) {
    switch(keyCode) {
    case 37:  // left
    case 72:  // h
        return ACTION_LEFT;

    case 40:  // down
    case 74:  // j
        return ACTION_DOWN;

    case 38:  // up
    case 75:  // k
        return ACTION_UP;

    case 39:  // right
    case 76:  // l
        return ACTION_RIGHT;

    case 88:  // x
        return ACTION_DELAY;

    case 90:  // z
        return ACTION_SHOT;
    }
    return 0;
}

function getSwitchByCharCode(charCode) {
    switch(charCode) {
    case 97:  // a
        return SWITCH_BOMB;
    case 99:  // c
        return SWITCH_BULLET;
    }
    return 0;
}

function setKeyDown(keyCode) {
    currentAction |= getActionByKeyCode(keyCode);
}

function setKeyUp(keyCode) {
    currentAction &= (0xffff - getActionByKeyCode(keyCode));
}

function isInAction(act) {
    return currentAction & act;
}

function setKeyPress(charCode) {
    currentSwitch |= getSwitchByCharCode(charCode);
}

function unsetSwitch(sw) {
    currentSwitch &= (0xffff - sw);
}

function isSetSwitch(sw) {
    return currentSwitch & sw;
}


/*
 *  Shot
 *    - update / switch barrages
 */
var Shot = function() {
    this.shot = function() {
    };
};

var ManualShot = function() {
    var that = new Shot();

    that.index = 0;
    that.shot  = function(trooper, enemy) {
        if (isSetSwitch(SWITCH_BULLET)) {
            if (++this.index >= trooper.barrages.length)
                this.index = 0;
            unsetSwitch(SWITCH_BULLET);
        }
        if (isSetSwitch(SWITCH_BOMB)) {
            trooper.addBomb();
            unsetSwitch(SWITCH_BOMB);
        }
        if (isInAction(ACTION_SHOT))
            trooper.barrages[this.index].update(trooper, enemy);
    };

    return that;
};

var BlazeRandomShot = function() {
    var that = new Shot();

    that.index = 0;
    that.shot  = function(trooper, enemy) {
        if (trooper.barrages[this.index].update(trooper, enemy))
            this.index = Math.floor(Math.random() * trooper.barrages.length);
    };

    return that;
};

var BlazeRotateShot = function(count) {
    var that = new BlazeRandomShot();

    if (!count)
        count = 0;

    that.i    = 0;
    that.shot = function(trooper, enemy) {
        if (trooper.barrages[this.index].update(trooper, enemy)) {
            if (++this.i > count) {
                this.index++;
                this.i = 0;
            }
            if (this.index >= trooper.barrages.length)
                this.index = 0;
        }
    };

    return that;
};

var BlazeRotateShot5 = function() {
    return new BlazeRotateShot(5);
};

var BlazeRotateShot10 = function() {
    return new BlazeRotateShot(10);
};


/*
 *  Action
 *    - call shot / move trooper
 */
var Action = function(shot) {
    this.shot   = shot;
    this.update = function(trooper, enemy) {
        if (trooper.isDead())
            return;
        this.shot.shot(trooper, enemy);
        return this.move(trooper, enemy);
    };
    this.move = function(trooper, enemy) {
        return true;
    };
};

var ManualAction = function(shot) {
    var that = new Action(shot);

    that.move = function(trooper, enemy) {
        var x = 0;
        var y = 0;
        var s = trooper.speed;

        if (isInAction(ACTION_DELAY))
            s /= 2;
        if (isInAction(ACTION_RIGHT))
            x += s;
        if (isInAction(ACTION_LEFT))
            x -= s;
        if (isInAction(ACTION_UP))
            y -= s;
        if (isInAction(ACTION_DOWN))
            y += s;

        trooper.move(x, y);

        return true;
    };

    return that;
};

var BezierAction = function(shot, resolution) {
    that = new Action(shot);

    if (!resolution)
        resolution = 1500;

    that.count      = 0;
    that.resolution = 0;

    that.move = function(trooper, enemy) {
        if (this.count >= this.resolution) {
            this.count      = 0;
            this.resolution = resolution / trooper.speed;

            this.p0 = this.getPoint0(trooper, enemy);
            this.p1 = this.getPoint1(trooper, enemy);
            this.p2 = this.getPoint2(trooper, enemy);
            this.p3 = this.getPoint3(trooper, enemy);
        }

        var t  = this.count++ / this.resolution;
        var t1 = 1 - t;
        var p0 = Math.pow(t1, 3);
        var p1 = 3 * t * Math.pow(t1, 2);
        var p2 = 3 * Math.pow(t, 2) * t1;
        var p3 = Math.pow(t, 3);

        trooper.moveAbs(
            this.p0[0] * p0 + this.p1[0] * p1 +
                this.p2[0] * p2 + this.p3[0] * p3,
            this.p0[1] * p0 + this.p1[1] * p1 +
                this.p2[1] * p2 + this.p3[1] * p3
        );

        if (this.count >= this.resolution)
            return true;

        return false;
    };

    return that;
};

var TriangleAction = function(shot, resolution) {
    var that = new BezierAction(shot, resolution);

    that.getPoint0 = function(trooper, enemy) {
        return [trooper.x, trooper.y];
    };
    that.getPoint1 = function(trooper, enemy) {
        return [trooper.x - (trooper.speed * 10),
                trooper.y + (trooper.speed * 10)];
    };
    that.getPoint2 = function(trooper, enemy) {
        return [trooper.x + (trooper.speed * 10),
                trooper.y + (trooper.speed * 10)];
    };
    that.getPoint3 = function(trooper, enemy) {
        return [trooper.x, trooper.y];
    };

    return that;
};

var RandomAction = function(shot, resolution) {
    var that = new BezierAction(shot, resolution);

    that.getPoint0 = function(trooper, enemy) {
        return [trooper.x, trooper.y];
    };
    that.getPoint1 = function(trooper, enemy) {
        return [Math.floor(Math.random() * trooper.w),
                Math.floor(Math.random() * trooper.h)];
    };
    that.getPoint2 = function(trooper, enemy) {
        return [Math.floor(Math.random() * trooper.w),
                Math.floor(Math.random() * trooper.h)];
    };
    that.getPoint3 = function(trooper, enemy) {
        return [Math.floor(Math.random() * trooper.w),
                Math.floor(Math.random() * trooper.h)];
    };

    return that;
};

var RandomReturnAction = function(shot, resolution) {
    var that = new RandomAction(shot, resolution);

    that.getPoint3 = function(trooper, enemy) {
        return [Math.floor(Math.random() * trooper.w), trooper.y];
    };

    return that;
};

var RandomStokingAction = function(shot, resolution) {
    var that = new RandomAction(shot, resolution);

    that.getPoint3 = function(trooper, enemy) {
        return [enemy.x, Math.floor(Math.random() * trooper.h)];
    };

    return that;
};

var RandomStokingReturnAction = function(shot, resolution) {
    var that = new RandomAction(shot, resolution);

    that.getPoint3 = function(trooper, enemy) {
        return [enemy.x, trooper.y];
    };

    return that;
};


/*
 *  ActionList
 *    - call / switch actions
 */
var ActionList = function(list) {
    this.list   = list;
    this.index  = 0;
    this.update = function(trooper, enemy) {
        this.updateIdx(this.list[this.index].update(trooper, enemy),
                       trooper, enemy);
    };
    this.updateIdx = function(end, trooper, enemy) {
        this.index = 0;
    };
};

var RandomActionList = function(list) {
    var that = new ActionList(list);

    that.updateIdx = function(end, trooper, enemy) {
        if (end)
            this.index = Math.floor(Math.random() * this.list.length);
    };

    return that;
};

var RotateActionList = function(list, count) {
    var that = new ActionList(list);

    if (!count)
        count = 0;

    that.i         = 0;
    that.updateIdx = function(end, trooper, enemy) {
        if (end) {
            if (++this.i > count) {
                this.index++;
                this.i = 0;
            }
            if (this.index >= this.list.length)
                this.index = 0;
        }
    };

    return that;
};

var RotateActionList5 = function(list) {
    return new RotateActionList(list, 5);
};

var RotateActionList10 = function(list) {
    return new RotateActionList(list, 10);
};

var DamageRandomActionList = function(list) {
    var that = new ActionList(list);

    that.life      = 0;
    that.updateIdx = function(end, trooper, enemy) {
        if (!this.life)
            this.life = trooper.life;

        if (end && trooper.life != this.life) {
            this.index = Math.floor(Math.random() * this.list.length);
            this.life  = trooper.life;
        }
    };

    return that;
};

var DamageRotateActionList = function(list) {
    var that = new ActionList(list);

    that.life      = 0;
    that.updateIdx = function(end, trooper, enemy) {
        if (!this.life)
            this.life = trooper.life;

        if (end && trooper.life != this.life) {
            if (++this.index >= this.list.length)
                this.index = 0;

            this.life = trooper.life;
        }
    };

    return that;
};


var EnemyShots       = [BlazeRandomShot, BlazeRotateShot,
                        BlazeRotateShot5, BlazeRotateShot10];
var EnemyActions     = [TriangleAction, RandomAction,
                        RandomReturnAction, RandomStokingAction,
                        RandomStokingReturnAction];
var EnemyActionLists = [RandomActionList, RotateActionList,
                        RotateActionList5, RotateActionList10,
                        DamageRandomActionList, DamageRotateActionList];
