/* -*- coding: utf-8 -*-
 *
 * Copyright (C) 2010 by Atzm WATANABE <atzm@atzm.org>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License (version 2) as
 *  published by the Free Software Foundation.  It is distributed in the
 *  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 * $Id$
 *
 */

var Barrage = function(bullet, size, color, interval, speed, way, dir) {
    this.bullet   = bullet;
    this.size     = size;
    this.color    = color;
    this.interval = interval;
    this.speed    = speed;
    this.way      = way;
    this.dir      = dir;

    this.aim = function(trooper, enemy) {
        var dx = enemy.x - trooper.x;
        var dy = enemy.y - trooper.y;
        return Math.atan2(dy, dx);
    };
    this.angle = function(trooper, enemy) {
        if (this.dir != null)
            return this.dir;
        return this.aim(trooper, enemy) / Math.PI;
    };
    this.update = function(trooper, enemy) {
        return false;
    };
    this.posX = function() {
        return 0;
    };
    this.posY = function() {
        return 0;
    };
};

var LinerBarrage = function(bullet, size, color, interval, speed, way, dir) {
    var that = new Barrage(bullet, size, color, interval, speed, way, dir);

    that.state  = 0;
    that.update = function(trooper, enemy) {
        if (++this.state < this.interval)
            return false;

        var dir = this.angle(trooper, enemy);

        trooper.addBullet(this.bullet, this.size, this.color,
                          this.posX(), this.posY(),
                          dir * Math.PI, this.speed);

        for (var i = 2; i < this.way + 1; i++) {
            var d = 0.5 / i;

            trooper.addBullet(this.bullet, this.size, this.color,
                              this.posX(), this.posY(),
                              (dir - d) * Math.PI, this.speed);
            trooper.addBullet(this.bullet, this.size, this.color,
                              this.posX(), this.posY(),
                              (dir + d) * Math.PI, this.speed);
        }

        this.state = 0;
        return true;
    };

    return that;
};

var ArchBarrage = function(bullet, size, color,
                           interval, speed, way, dir, delta) {

    var that = new Barrage(bullet, size, color, interval, speed, way, dir);

    that.delta = delta;

    if (that.way % 2)
        that.way++;

    if (that.delta == null)
        that.delta = 0.5 / that.way;

    that.state  = 0;
    that.update = function(trooper, enemy) {
        if (++this.state < this.interval)
            return false;

        var angle = this.angle(trooper, enemy) - (this.delta * this.way / 2);

        for (var i = 0; i < this.way; i++) {
            trooper.addBullet(this.bullet, this.size, this.color,
                              this.posX(), this.posY(),
                              angle * Math.PI, this.speed);
            angle += this.delta;
        }

        this.state = 0;
        return true;
    };

    return that;
};

var CircularBarrage = function(bullet, size, color,
                               interval, speed, way, dir) {
    if (way % 2)
        way++;

    var that = new ArchBarrage(bullet, size, color,
                               interval, speed, way, dir, 2 / way);
    return that;
}

var DelayedArchBarrage = function(bullet, size, color,
                                  interval, speed, way, dir, delta) {

    var that = new Barrage(bullet, size, color, interval, speed, way, dir);

    that.delta = delta;

    if (that.way % 2)
        that.way++;

    if (that.delta == null)
        that.delta = 0.5 / that.way;

    that.state    = 0;
    that.remain   = that.way;
    that.curAngle = null;

    that.update = function(trooper, enemy) {
        if (++this.state < this.interval)
            return false;

        if (this.curAngle == null)
            this.curAngle = this.angle(trooper, enemy) -
                this.delta * (this.way / 2);

        trooper.addBullet(this.bullet, this.size, this.color,
                          this.posX(), this.posY(),
                          this.curAngle * Math.PI, this.speed);

        this.curAngle += this.delta;

        if (--this.remain <= 0) {
            return this.reset();
        }

        return false;
    };
    that.reset = function() {
        this.state    = 0;
        this.remain   = this.way;
        this.curAngle = null;
        return true;
    };

    return that;
};

var DelayedCircularBarrage = function(bullet, size, color,
                                      interval, speed, way, dir) {
    if (way % 2)
        way++;

    var that = new DelayedArchBarrage(bullet, size, color,
                                      interval, speed, way, dir, 2 / way);
    return that;
};

var DelayedRecursiveArchBarrage = function(bullet, size, color,
                                           interval, speed, way, dir, cnt) {

    var that = new DelayedArchBarrage(bullet, size, color,
                                      interval, speed, way, dir);

    that.i   = 0;
    that.cnt = cnt;

    if (that.cnt == null)
        that.cnt = that.way / that.speed;

    that.reset = function() {
        if (++this.i < this.cnt) {
            this.remain  = this.way;
            this.delta  *= -1;
            return false;
        }
        this.state    = 0;
        this.remain   = this.way;
        this.curAngle = null;
        this.i        = 0;
        return true;
    };
    return that;
};

var DelayedRecursiveCircularBarrage = function(bullet, size, color,
                                               interval, speed, way, dir, cnt) {

    var that = new DelayedCircularBarrage(bullet, size, color,
                                          interval, speed, way, dir);

    that.i   = 0;
    that.cnt = cnt;

    if (that.cnt == null)
        that.cnt = that.way / that.speed;

    that.reset = function() {
        if (++this.i < this.cnt) {
            this.remain = this.way;
            return false;
        }
        this.state    = 0;
        this.remain   = this.way;
        this.curAngle = null;
        this.i        = 0;
        return true;
    };
    return that;
};

var MultiRounderBarrage = function(bullet, size, color,
                                   interval, speed, way, dir,
                                   klass, radius, num) {
    if (radius == null) {
        radius = size * way;
        if (radius > 50)
            radius = 50;
    }

    if (num == null)
        num = Math.ceil((1 / way) * 50);

    var start = 0.25;

    this.initBarrage = function() {
        this.brrg = new Array();
        this.flag = new Array();

        var w = Math.ceil(way / num);

        for (var i = 0; i < num; i++) {
            var b = new klass(bullet, size, color,
                              interval, speed, w, dir);

            b.r = radius;
            b.posX = function() {
                start += 1 / (num * 2);
                this.a = i * (1 / num) + start;
                return this.r * Math.cos(2 * Math.PI * this.a);
            };

            b.posY = function() {
                return this.r * Math.sin(2 * Math.PI * this.a);
            };

            this.brrg.push(b);
            this.flag.push(false);
        }
    };

    this.update = function(trooper, enemy) {
        for (var i = 0; i < this.brrg.length; i++) {
            if (!this.flag[i])
                this.flag[i] = this.brrg[i].update(trooper, enemy);
        }
        for (var i = 0; i < this.flag.length; i++) {
            if (!this.flag[i])
                return false;
        }
        this.initBarrage();
        return true;
    };

    this.initBarrage();
};

var MultiRounderArchBarrage = function(bullet, size, color,
                                       interval, speed, way, dir) {
    var that = new MultiRounderBarrage(bullet, size, color, interval, speed,
                                       way, dir, ArchBarrage);
    return that;
};

var MultiRounderCircularBarrage = function(bullet, size, color,
                                           interval, speed, way, dir) {
    var that = new MultiRounderBarrage(bullet, size, color, interval, speed,
                                       way, dir, CircularBarrage);
    return that;
};

var MultiRounderDelayedArchBarrage = function(bullet, size, color,
                                              interval, speed, way, dir) {
    var that = new MultiRounderBarrage(bullet, size, color, interval, speed,
                                       way, dir, DelayedArchBarrage);
    return that;
};

var MultiRounderDelayedCircularBarrage = function(bullet, size, color,
                                                  interval, speed, way, dir) {
    var that = new MultiRounderBarrage(bullet, size, color, interval, speed,
                                       way, dir, DelayedCircularBarrage);
    return that;
};

var MultiRounderDelayedRecursiveArchBarrage = function(bullet, size, color,
                                                       interval, speed, way, dir) {
    var that = new MultiRounderBarrage(bullet, size, color, interval, speed,
                                       way, dir, DelayedRecursiveArchBarrage);
    return that;
};

var MultiRounderDelayedRecursiveCircularBarrage = function(bullet, size, color,
                                                           interval, speed, way, dir) {
    var that = new MultiRounderBarrage(bullet, size, color, interval, speed,
                                       way, dir, DelayedRecursiveCircularBarrage);
    return that;
};


var EnemyBarrages = [LinerBarrage,
                     ArchBarrage,
                     CircularBarrage,
                     DelayedArchBarrage,
                     DelayedCircularBarrage,
                     DelayedRecursiveArchBarrage,
                     DelayedRecursiveCircularBarrage,
                     MultiRounderArchBarrage,
                     MultiRounderCircularBarrage,
                     MultiRounderDelayedArchBarrage,
                     MultiRounderDelayedCircularBarrage,
                     MultiRounderDelayedRecursiveArchBarrage,
                     MultiRounderDelayedRecursiveCircularBarrage];
